const path = require('path');

// ============================================================================
// APPLICATION CONFIGURATION
// ============================================================================

const config = {
  // Application Info
  APP_NAME: 'Ayurwell Enterprise Platform',
  APP_VERSION: '1.0.0',
  
  // Server Configuration
  PORT: process.env.PORT || 44322, // webd.pl assigned port
  JSON_LIMIT: '10mb',
  NODE_ENV: process.env.NODE_ENV || 'production',
  
  // Database Configuration
  DB_PATH: path.join(__dirname, '..', 'ayurwell_platform.sqlite'),
  
  // Security Configuration
  JWT_SECRET: process.env.JWT_SECRET || 'AyurwellSecret2024!',
  JWT_EXPIRES_IN: '7d',
  BCRYPT_SALT_ROUNDS: 10,
  
  // CORS Configuration for webd.pl hosting
  CORS_ORIGINS: [
    'https://ayurwell.world',
    'https://www.ayurwell.world',
    'http://localhost:3000', // For development
    'http://localhost:3001'
  ],
  
  // Rate Limiting
  RATE_LIMIT_WINDOW_MS: 15 * 60 * 1000, // 15 minutes
  RATE_LIMIT_MAX: 100, // limit each IP to 100 requests per windowMs
  AUTH_RATE_LIMIT_MAX: 50, // limit auth endpoints to 5 requests per windowMs
  
  // File Upload Configuration
  UPLOAD_PATH: path.join(__dirname, '..', 'uploads'),
  MAX_FILE_SIZE: 5 * 1024 * 1024, // 5MB
  
  // Application Features
  FEATURES: {
    USER_REGISTRATION: true,
    CRM_MODULE: true,
    CLIENT_DASHBOARD: true,
    FILE_UPLOADS: true
  },
  
  // Development flags
  IS_DEVELOPMENT: process.env.NODE_ENV === 'development',
  IS_PRODUCTION: process.env.NODE_ENV === 'production',
  
  // Logging Configuration
  LOG_LEVEL: process.env.LOG_LEVEL || 'info',
  LOG_TO_FILE: false, // Set to true if you want file logging
  
  // Session Configuration
  SESSION_SECRET: process.env.SESSION_SECRET || 'Ayurwell_Session_Secret_2024!',
  
  // API Configuration
  API_PREFIX: '/api',
  API_VERSION: 'v1'
};

// ============================================================================
// ENVIRONMENT VALIDATION
// ============================================================================

// Validate required environment variables in production
if (config.IS_PRODUCTION) {
  const requiredEnvVars = [];
  
  // Check if JWT_SECRET is using default (warn only)
  if (config.JWT_SECRET === 'TwojSekretnyKlucz_Production_2024!') {
    console.warn('⚠️  WARNING: Using default JWT_SECRET. Set JWT_SECRET environment variable for production.');
  }
  
  // Validate required variables
  requiredEnvVars.forEach(envVar => {
    if (!process.env[envVar]) {
      console.error(`❌ Required environment variable ${envVar} is not set`);
      process.exit(1);
    }
  });
}

// ============================================================================
// UTILITY FUNCTIONS
// ============================================================================

/**
 * Get configuration value safely
 * @param {string} key - Configuration key
 * @param {*} defaultValue - Default value if key not found
 * @returns {*} Configuration value
 */
config.get = function(key, defaultValue = null) {
  return this[key] !== undefined ? this[key] : defaultValue;
};

/**
 * Check if feature is enabled
 * @param {string} feature - Feature name
 * @returns {boolean} True if feature is enabled
 */
config.isFeatureEnabled = function(feature) {
  return this.FEATURES[feature] === true;
};

/**
 * Get database configuration object
 * @returns {object} Database configuration
 */
config.getDatabaseConfig = function() {
  return {
    path: this.DB_PATH,
    options: {
      busyTimeout: 30000, // 30 seconds
      verbose: this.IS_DEVELOPMENT ? console.log : null
    }
  };
};

/**
 * Get CORS configuration object
 * @returns {object} CORS configuration
 */
config.getCorsConfig = function() {
  return {
    origin: this.CORS_ORIGINS,
    credentials: true,
    methods: ['GET', 'POST', 'PUT', 'DELETE', 'OPTIONS', 'PATCH'],
    allowedHeaders: [
      'Content-Type', 
      'Authorization', 
      'X-Requested-With',
      'Accept',
      'Origin'
    ],
    exposedHeaders: ['X-Total-Count', 'X-Page-Count'],
    optionsSuccessStatus: 200 // Some legacy browsers choke on 204
  };
};

// ============================================================================
// EXPORT CONFIGURATION
// ============================================================================

module.exports = config;