const express = require('express');
const database = require('../models/database');
const { 
  verifyToken, 
  requireCrmAccess, 
  validateRequiredFields, 
  sanitizeInput 
} = require('../middleware/auth');

const router = express.Router();

// Apply authentication to all CRM routes
router.use(verifyToken);
router.use(requireCrmAccess);

// ============================================================================
// LEADS MANAGEMENT
// ============================================================================

/**
 * GET /api/crm/leads
 * Get all leads with optional filtering and pagination
 */
router.get('/leads', async (req, res) => {
  try {
    const { 
      status, 
      assigned_to, 
      lead_source, 
      category,
      limit = 50, 
      offset = 0,
      search 
    } = req.query;
    
    let sql = `
      SELECT l.*, u.email as assigned_email 
      FROM crm_leads l
      LEFT JOIN users u ON l.assigned_to = u.id
      WHERE 1=1
    `;
    const params = [];
    
    // Add filters
    if (status) {
      sql += ' AND l.lead_status = ?';
      params.push(status);
    }
    
    if (assigned_to) {
      sql += ' AND l.assigned_to = ?';
      params.push(assigned_to);
    }
    
    if (lead_source) {
      sql += ' AND l.lead_source = ?';
      params.push(lead_source);
    }
    
    if (category) {
      sql += ' AND l.category = ?';
      params.push(category);
    }
    
    if (search) {
      sql += ' AND (l.company_name LIKE ? OR l.contact_name LIKE ? OR l.email LIKE ?)';
      const searchTerm = `%${search}%`;
      params.push(searchTerm, searchTerm, searchTerm);
    }
    
    sql += ' ORDER BY l.created_at DESC LIMIT ? OFFSET ?';
    params.push(parseInt(limit), parseInt(offset));
    
    const leads = await database.query(sql, params);
    
    // Get total count for pagination
    let countSql = 'SELECT COUNT(*) as total FROM crm_leads WHERE 1=1';
    const countParams = [];
    
    if (status) {
      countSql += ' AND lead_status = ?';
      countParams.push(status);
    }
    if (assigned_to) {
      countSql += ' AND assigned_to = ?';
      countParams.push(assigned_to);
    }
    if (lead_source) {
      countSql += ' AND lead_source = ?';
      countParams.push(lead_source);
    }
    if (category) {
      countSql += ' AND category = ?';
      countParams.push(category);
    }
    if (search) {
      countSql += ' AND (company_name LIKE ? OR contact_name LIKE ? OR email LIKE ?)';
      const searchTerm = `%${search}%`;
      countParams.push(searchTerm, searchTerm, searchTerm);
    }
    
    const countResult = await database.get(countSql, countParams);
    
    res.json({
      leads: leads,
      pagination: {
        total: countResult.total,
        limit: parseInt(limit),
        offset: parseInt(offset),
        has_more: countResult.total > (parseInt(offset) + parseInt(limit))
      }
    });
    
  } catch (error) {
    console.error('❌ Error fetching leads:', error);
    res.status(500).json({ 
      message: 'Error fetching leads',
      error: 'FETCH_LEADS_ERROR'
    });
  }
});

/**
 * GET /api/crm/leads/:id
 * Get a single lead by ID
 */
router.get('/leads/:id', async (req, res) => {
  try {
    const { id } = req.params;
    
    if (!id || isNaN(parseInt(id))) {
      return res.status(400).json({ 
        message: 'Invalid lead ID',
        error: 'INVALID_ID'
      });
    }
    
    const lead = await database.get(`
      SELECT l.*, u.email as assigned_email 
      FROM crm_leads l
      LEFT JOIN users u ON l.assigned_to = u.id
      WHERE l.id = ?
    `, [id]);
    
    if (!lead) {
      return res.status(404).json({ 
        message: 'Lead not found',
        error: 'LEAD_NOT_FOUND'
      });
    }
    
    // Get recent activities for this lead
    const activities = await database.query(
      'SELECT * FROM crm_activities WHERE lead_id = ? ORDER BY scheduled_date DESC LIMIT 10',
      [id]
    );
    
    res.json({
      lead: lead,
      recent_activities: activities
    });
    
  } catch (error) {
    console.error('❌ Error fetching lead:', error);
    res.status(500).json({ 
      message: 'Error fetching lead',
      error: 'FETCH_LEAD_ERROR'
    });
  }
});

/**
 * POST /api/crm/leads
 * Create a new lead
 */
router.post('/leads',
  sanitizeInput,
  validateRequiredFields(['company_name']),
  async (req, res) => {
    try {
      const {
        company_name,
        contact_name,
        phone,
        email,
        website,
        lead_source,
        lead_status = 'new',
        category,
        address,
        city,
        notes,
        assigned_to
      } = req.body;

      // Validate assigned_to if provided
      if (assigned_to) {
        const assignedUser = await database.get(
          'SELECT id FROM users WHERE id = ? AND crm_access = 1',
          [assigned_to]
        );
        if (!assignedUser) {
          return res.status(400).json({
            message: 'Invalid assigned user',
            error: 'INVALID_ASSIGNED_USER'
          });
        }
      }

      const result = await database.run(`
        INSERT INTO crm_leads (
          company_name, contact_name, phone, email, website,
          lead_source, lead_status, category, address, city,
          notes, assigned_to
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
      `, [
        company_name,
        contact_name || null,
        phone || null,
        email || null,
        website || null,
        lead_source || null,
        lead_status,
        category || null,
        address || null,
        city || null,
        notes || null,
        assigned_to || null
      ]);

      if (!result.lastID) {
        throw new Error('Failed to create lead');
      }

      // Get the created lead with assigned user info
      const newLead = await database.get(`
        SELECT l.*, u.email as assigned_email 
        FROM crm_leads l
        LEFT JOIN users u ON l.assigned_to = u.id
        WHERE l.id = ?
      `, [result.lastID]);

      console.log('✅ Lead created:', company_name);

      res.status(201).json({
        message: 'Lead created successfully',
        lead: newLead
      });

    } catch (error) {
      console.error('❌ Error creating lead:', error);
      res.status(500).json({
        message: 'Error creating lead',
        error: 'CREATE_LEAD_ERROR'
      });
    }
  }
);

/**
 * PUT /api/crm/leads/:id
 * Update an existing lead
 */
router.put('/leads/:id',
  sanitizeInput,
  async (req, res) => {
    try {
      const { id } = req.params;
      
      if (!id || isNaN(parseInt(id))) {
        return res.status(400).json({ 
          message: 'Invalid lead ID',
          error: 'INVALID_ID'
        });
      }

      // Check if lead exists
      const existingLead = await database.get('SELECT id FROM crm_leads WHERE id = ?', [id]);
      if (!existingLead) {
        return res.status(404).json({
          message: 'Lead not found',
          error: 'LEAD_NOT_FOUND'
        });
      }

      const {
        company_name,
        contact_name,
        phone,
        email,
        website,
        lead_source,
        lead_status,
        category,
        address,
        city,
        notes,
        assigned_to
      } = req.body;

      // Validate assigned_to if provided
      if (assigned_to) {
        const assignedUser = await database.get(
          'SELECT id FROM users WHERE id = ? AND crm_access = 1',
          [assigned_to]
        );
        if (!assignedUser) {
          return res.status(400).json({
            message: 'Invalid assigned user',
            error: 'INVALID_ASSIGNED_USER'
          });
        }
      }

      const result = await database.run(`
        UPDATE crm_leads SET
          company_name = COALESCE(?, company_name),
          contact_name = COALESCE(?, contact_name),
          phone = COALESCE(?, phone),
          email = COALESCE(?, email),
          website = COALESCE(?, website),
          lead_source = COALESCE(?, lead_source),
          lead_status = COALESCE(?, lead_status),
          category = COALESCE(?, category),
          address = COALESCE(?, address),
          city = COALESCE(?, city),
          notes = COALESCE(?, notes),
          assigned_to = COALESCE(?, assigned_to),
          updated_at = CURRENT_TIMESTAMP
        WHERE id = ?
      `, [
        company_name,
        contact_name,
        phone,
        email,
        website,
        lead_source,
        lead_status,
        category,
        address,
        city,
        notes,
        assigned_to,
        id
      ]);

      if (result.changes === 0) {
        return res.status(404).json({
          message: 'Lead not found',
          error: 'LEAD_NOT_FOUND'
        });
      }

      // Get updated lead
      const updatedLead = await database.get(`
        SELECT l.*, u.email as assigned_email 
        FROM crm_leads l
        LEFT JOIN users u ON l.assigned_to = u.id
        WHERE l.id = ?
      `, [id]);

      console.log('✅ Lead updated:', id);

      res.json({
        message: 'Lead updated successfully',
        lead: updatedLead
      });

    } catch (error) {
      console.error('❌ Error updating lead:', error);
      res.status(500).json({
        message: 'Error updating lead',
        error: 'UPDATE_LEAD_ERROR'
      });
    }
  }
);

/**
 * DELETE /api/crm/leads/:id
 * Delete a lead
 */
router.delete('/leads/:id', async (req, res) => {
  try {
    const { id } = req.params;
    
    if (!id || isNaN(parseInt(id))) {
      return res.status(400).json({ 
        message: 'Invalid lead ID',
        error: 'INVALID_ID'
      });
    }

    // Check if lead exists
    const lead = await database.get('SELECT id, company_name FROM crm_leads WHERE id = ?', [id]);
    if (!lead) {
      return res.status(404).json({
        message: 'Lead not found',
        error: 'LEAD_NOT_FOUND'
      });
    }

    // Delete associated activities first
    await database.run('DELETE FROM crm_activities WHERE lead_id = ?', [id]);
    
    // Delete the lead
    const result = await database.run('DELETE FROM crm_leads WHERE id = ?', [id]);

    if (result.changes === 0) {
      throw new Error('Failed to delete lead');
    }

    console.log('✅ Lead deleted:', lead.company_name);

    res.json({
      message: 'Lead deleted successfully',
      deleted_lead: { id: parseInt(id), company_name: lead.company_name }
    });

  } catch (error) {
    console.error('❌ Error deleting lead:', error);
    res.status(500).json({
      message: 'Error deleting lead',
      error: 'DELETE_LEAD_ERROR'
    });
  }
});

// ============================================================================
// ACTIVITIES MANAGEMENT
// ============================================================================

/**
 * GET /api/crm/activities
 * Get all activities with optional filtering
 */
router.get('/activities', async (req, res) => {
  try {
    const { 
      lead_id, 
      activity_type, 
      completed,
      limit = 50, 
      offset = 0 
    } = req.query;
    
    let sql = `
      SELECT la.*, l.company_name as lead_company_name, u.email as completed_by_email
      FROM crm_activities la
      LEFT JOIN crm_leads l ON la.lead_id = l.id
      LEFT JOIN users u ON la.completed_by = u.id
      WHERE 1=1
    `;
    const params = [];
    
    if (lead_id) {
      sql += ' AND la.lead_id = ?';
      params.push(lead_id);
    }
    
    if (activity_type) {
      sql += ' AND la.activity_type = ?';
      params.push(activity_type);
    }
    
    if (completed !== undefined) {
      if (completed === 'true') {
        sql += ' AND la.completed_date IS NOT NULL';
      } else {
        sql += ' AND la.completed_date IS NULL';
      }
    }
    
    sql += ' ORDER BY la.scheduled_date ASC LIMIT ? OFFSET ?';
    params.push(parseInt(limit), parseInt(offset));
    
    const activities = await database.query(sql, params);
    
    res.json(activities);
    
  } catch (error) {
    console.error('❌ Error fetching activities:', error);
    res.status(500).json({ 
      message: 'Error fetching activities',
      error: 'FETCH_ACTIVITIES_ERROR'
    });
  }
});

/**
 * POST /api/crm/activities
 * Create a new activity
 */
router.post('/activities',
  sanitizeInput,
  validateRequiredFields(['lead_id', 'activity_type', 'description']),
  async (req, res) => {
    try {
      const { 
        lead_id, 
        activity_type, 
        description, 
        scheduled_date, 
        notes 
      } = req.body;

      // Validate lead exists
      const lead = await database.get('SELECT id FROM crm_leads WHERE id = ?', [lead_id]);
      if (!lead) {
        return res.status(400).json({
          message: 'Lead not found',
          error: 'LEAD_NOT_FOUND'
        });
      }

      const result = await database.run(`
        INSERT INTO crm_activities (
          lead_id, activity_type, description, scheduled_date, notes
        ) VALUES (?, ?, ?, ?, ?)
      `, [
        lead_id,
        activity_type,
        description,
        scheduled_date || null,
        notes || null
      ]);

      if (!result.lastID) {
        throw new Error('Failed to create activity');
      }

      // Get the created activity with lead info
      const newActivity = await database.get(`
        SELECT la.*, l.company_name as lead_company_name
        FROM crm_activities la
        JOIN crm_leads l ON la.lead_id = l.id
        WHERE la.id = ?
      `, [result.lastID]);

      console.log('✅ Activity created for lead:', lead_id);

      res.status(201).json({
        message: 'Activity created successfully',
        activity: newActivity
      });

    } catch (error) {
      console.error('❌ Error creating activity:', error);
      res.status(500).json({
        message: 'Error creating activity',
        error: 'CREATE_ACTIVITY_ERROR'
      });
    }
  }
);

/**
 * PUT /api/crm/activities/:id/complete
 * Mark an activity as completed
 */
router.put('/activities/:id/complete', async (req, res) => {
  try {
    const { id } = req.params;
    const { notes } = req.body;
    
    if (!id || isNaN(parseInt(id))) {
      return res.status(400).json({ 
        message: 'Invalid activity ID',
        error: 'INVALID_ID'
      });
    }

    const result = await database.run(`
      UPDATE crm_activities SET
        completed_date = CURRENT_TIMESTAMP,
        notes = CASE WHEN ? IS NOT NULL THEN ? ELSE notes END,
        completed_by = ?
      WHERE id = ?
    `, [notes, notes, req.user.id, id]);

    if (result.changes === 0) {
      return res.status(404).json({ 
        message: 'Activity not found',
        error: 'ACTIVITY_NOT_FOUND'
      });
    }

    console.log('✅ Activity completed:', id);
    res.json({ 
      message: 'Activity marked as completed',
      activity: { id: parseInt(id) }
    });
    
  } catch (error) {
    console.error('❌ Error updating activity:', error);
    res.status(500).json({ 
      message: 'Error updating activity',
      error: 'UPDATE_ACTIVITY_ERROR'
    });
  }
});

// ============================================================================
// SALES TEAM MANAGEMENT
// ============================================================================

/**
 * GET /api/crm/sales-team
 * Get all sales team members
 */
router.get('/sales-team', async (req, res) => {
  try {
    const team = await database.query(`
      SELECT st.id, st.position, st.hire_date, u.id as user_id, u.email, u.first_name, u.last_name
      FROM crm_sales_team st
      JOIN users u ON st.user_id = u.id
      WHERE u.active = 1
      ORDER BY st.hire_date DESC
    `);
    
    res.json(team);
    
  } catch (error) {
    console.error('❌ Error fetching sales team:', error);
    res.status(500).json({ 
      message: 'Error fetching sales team',
      error: 'FETCH_TEAM_ERROR'
    });
  }
});

/**
 * POST /api/crm/sales-team
 * Add a new team member
 */
router.post('/sales-team',
  sanitizeInput,
  validateRequiredFields(['user_id', 'position']),
  async (req, res) => {
    try {
      const { user_id, position, hire_date } = req.body;

      // Check if user exists and has CRM access
      const user = await database.get(
        'SELECT id, email FROM users WHERE id = ? AND crm_access = 1',
        [user_id]
      );
      if (!user) {
        return res.status(400).json({ 
          message: 'User does not exist or lacks CRM access',
          error: 'USER_NOT_FOUND'
        });
      }
      
      // Check if user is already a team member
      const existingMember = await database.get(
        'SELECT id FROM crm_sales_team WHERE user_id = ?',
        [user_id]
      );
      if (existingMember) {
        return res.status(400).json({ 
          message: 'This user is already a team member',
          error: 'USER_ALREADY_MEMBER'
        });
      }
      
      const finalHireDate = hire_date || new Date().toISOString();
      
      const result = await database.run(
        'INSERT INTO crm_sales_team (user_id, position, hire_date) VALUES (?, ?, ?)',
        [user_id, position, finalHireDate]
      );

      if (!result.lastID) {
        throw new Error('Failed to add team member');
      }

      // Get the created team member with user info
      const newMember = await database.get(`
        SELECT st.id, st.position, st.hire_date, u.id as user_id, u.email, u.first_name, u.last_name
        FROM crm_sales_team st
        JOIN users u ON st.user_id = u.id
        WHERE st.id = ?
      `, [result.lastID]);

      console.log('✅ Team member added:', user.email);

      res.status(201).json({
        message: 'Team member added successfully',
        team_member: newMember
      });

    } catch (error) {
      console.error('❌ Error adding team member:', error);
      res.status(500).json({
        message: 'Error adding team member',
        error: 'ADD_TEAM_MEMBER_ERROR'
      });
    }
  }
);

/**
 * DELETE /api/crm/sales-team/:id
 * Remove a team member
 */
router.delete('/sales-team/:id', async (req, res) => {
  try {
    const { id } = req.params;
    
    if (!id || isNaN(parseInt(id))) {
      return res.status(400).json({ 
        message: 'Invalid team member ID',
        error: 'INVALID_ID'
      });
    }

    // Get team member info before deleting
    const member = await database.get(`
      SELECT st.id, u.email 
      FROM crm_sales_team st
      JOIN users u ON st.user_id = u.id
      WHERE st.id = ?
    `, [id]);

    if (!member) {
      return res.status(404).json({
        message: 'Team member not found',
        error: 'TEAM_MEMBER_NOT_FOUND'
      });
    }

    const result = await database.run('DELETE FROM crm_sales_team WHERE id = ?', [id]);

    if (result.changes === 0) {
      throw new Error('Failed to remove team member');
    }

    console.log('✅ Team member removed:', member.email);

    res.json({
      message: 'Team member removed successfully',
      removed_member: { id: parseInt(id), email: member.email }
    });

  } catch (error) {
    console.error('❌ Error removing team member:', error);
    res.status(500).json({
      message: 'Error removing team member',
      error: 'REMOVE_TEAM_MEMBER_ERROR'
    });
  }
});

// ============================================================================
// USERS MANAGEMENT (for CRM)
// ============================================================================

/**
 * GET /api/crm/users
 * Get all CRM users (for assignments, team management, etc.)
 */
router.get('/users', async (req, res) => {
  try {
    const users = await database.query(`
      SELECT id, email, first_name, last_name, crm_access, created_at, last_login
      FROM users 
      WHERE crm_access = 1 AND active = 1
      ORDER BY email
    `);
    
    res.json(users);
    
  } catch (error) {
    console.error('❌ Error fetching CRM users:', error);
    res.status(500).json({ 
      message: 'Error fetching users',
      error: 'FETCH_USERS_ERROR'
    });
  }
});

// ============================================================================
// EXPORTS
// ============================================================================

module.exports = router;